class CookieComplianceBanner {


  static instances = new Map()
  static selector = '[data-ccb]'
  static DISMISSED_KEY = 'ccb_dismissed'

  static load() {
    const elements = document.querySelectorAll(CookieComplianceBanner.selector)

    elements.forEach(element => {
      let options = JSON.parse(element.dataset.ccb || '{}' )
      let instance = new CookieComplianceBanner(element, options)

      CookieComplianceBanner.instances.set(element, instance)

      instance.init()
    })
  }


  element
  options

  constructor(element, options) {
    this.element = element
    this.options = this.#settings(options)
    console.log(this)
  }


  #settings(options) {
    return {
      close: this.element.querySelector(options.close),
      agree: this.element.querySelector(options.agree),
      privacy_policy_url: options.privacy_policy_url,
      auto_dismiss: options.auto_dismiss || false,
    }
  }

  get dismissed() {
    return sessionStorage.getItem(CookieComplianceBanner.DISMISSED_KEY) === '1'
  }

  #setInitialState() {
    if ( sessionStorage.getItem(CookieComplianceBanner.DISMISSED_KEY) === null ) {
      sessionStorage.setItem(CookieComplianceBanner.DISMISSED_KEY, '0')
    }

    this.dismissed
      ? this.dismiss()
      : setTimeout(() => this.element.classList.add('animate', 'is-animated'), 100)
  }

  #bindEvents() {
    this.options.close?.addEventListener('click', e => {
      e.preventDefault()
      this.dismiss()
    })
    this.options.agree?.addEventListener('click', e => {
      e.preventDefault()
      this.agree()
    })

    if ( this.options.auto_dismiss ) {
      ['keypress', 'mousedown', 'touchstart', 'scroll'].forEach(event => {
        document.addEventListener(event, () => {
          if ( this.dismissed ) return
          this.dismiss()
        })
      })
    }
  }

  #setupLinks() {
    try {
      let currentUrl = new URL(window.location.href)
      let targetUrl = new URL(this.options.privacy_policy_url, currentUrl.origin)
      let isExternal = currentUrl.origin !== targetUrl.origin

      document.querySelectorAll('[href*="#privacy-policy"]').forEach(link => {
        link.href = targetUrl.href
        link.target = isExternal ? '_blank' : '_self'
      })
    } catch (e) {
      console.error('The privacy policy url in Layout Settings is malformed.')
    }
  }

  dismiss() {
    sessionStorage.setItem(CookieComplianceBanner.DISMISSED_KEY, '1')
    this.element.remove()
  }

  agree() {
    this.dismiss()
  }

  init() {
    this.#bindEvents()
    this.#setInitialState()
    this.#setupLinks()
  }

}

export {
  CookieComplianceBanner
}
