<?php

namespace LeaseLeads\ThemeKit\Deploy\Commands;


use function LeaseLeads\ThemeKit\Core\Helpers\env;

abstract class BaseCommand {

    public array $arguments = [];

    public function set(string $name, $value) : void {
        $this->arguments[$name] = $value;
    }

    public function arguments() : array {
        return [];
    }

    abstract public function name() : string;

    abstract public function description() : string;

    abstract public function run() : void;

    protected function env() : array {
        $keys = [
            'STAGING_SITE_URL',
            'STAGING_SITE_PATH',
            'STAGING_SSH_USER',
        ];

        $vars = [];

        foreach($keys as $key) {
            if ( ! env($key) ) {
                throw new \Exception("Missing environment variable: {$key}");
            }

            $vars[$key] = env($key);
        }

        return $vars;
    }

    protected function ssh(): string
    {
        $vars = $this->env();
        return $vars['STAGING_SSH_USER'] . '@' . $vars['STAGING_SITE_URL'];
    }

    protected function localContent($path = '') : string {
        return WP_CONTENT_DIR . '/' . $path;
    }

    protected function remoteContent($path = '') : string {
        $vars = $this->env();
        return $vars['STAGING_SITE_PATH'] . '/wp-content/' . $path;
    }

    protected function rsyncDown($path, $options = '') : string {
        $remote_path = $this->remoteContent($path);
        $local_path = $this->localContent($path);
        $rsync_templates = 'rsync -rlptOz --delete --protocol=29 %s %s:%s %s';

        return sprintf(
            $rsync_templates,
            $options,
            $this->ssh(),
            $remote_path,
            $local_path
        );
    }

    protected function rsyncUp($path, $options = ''): string
    {
        $local_path = $this->localContent($path);
        $remote_path = $this->remoteContent($path);
        $rsync_templates = 'rsync -rlptOz --delete --protocol=29 %s %s %s:%s';

        return sprintf(
            $rsync_templates,
            $options,
            $local_path,
            $this->ssh(),
            $remote_path
        );
    }

    protected function wp($command, $flags = '', $context = 'remote') : string {
        $template = ($context === 'remote' ? 'wp ' : '') . '%s --skip-plugins --skip-themes %s';

        return sprintf(
            $template,
            $command,
            $flags
        );
    }

    protected function remoteCommand($command) : string {
        $remotewp = "cd {$this->env()['STAGING_SITE_PATH']}";
        return "ssh {$this->ssh()} \"{$remotewp}; {$command};\"";
    }

    protected function runCommand($command, $executor, $message) : void {
        \WP_CLI::line($message);
        \WP_CLI::line('======================================================================');
        \WP_CLI::line($command);
        $executor($command);
        \WP_CLI::line('');
        \WP_CLI::line('');
    }

}
