<?php

namespace LeaseLeads\ThemeKit\Deploy\Commands;

use LeaseLeads\ThemeKit\Core\Services\Filter;
use function LeaseLeads\ThemeKit\Core\Helpers\config;

class Push extends BaseCommand {

    public function name(): string
    {
        return 'push';
    }

    public function description(): string
    {
        return 'Pushes the database, plugins, and uploads to staging. Packages the theme for production use, and pushes it to staging.';
    }

    public function run(): void
    {
        $this->pushDatabase();
        $this->pushFiles();

        \WP_CLI::success('Database, uploads, themes, and plugins have been pushed to staging.');
    }

    protected function pushFiles() : void {
        $vars = $this->env();

        $themePath = get_template_directory();
        $themeDir = basename($themePath);

        $exclusions = Filter::apply(
            'deploy/commands/push/exclusions',
            config('deploy.ignores', []),
        );

        $exclusions = implode(' ', array_map(fn($ignore) => "--exclude={$ignore}", $exclusions));

        $commands = [
            [
                'command' => $this->rsyncUp('plugins/'),
                'executor' => fn($cmd) => exec($cmd),
                'message' => 'Pushing plugins to staging',
            ],
            [
                'command' => $this->rsyncUp('uploads/'),
                'executor' => fn($cmd) => exec($cmd),
                'message' => 'Pushing uploads to staging',
            ],
            [
                'command' => $this->wp('ll zip', '', 'remote'),
                'executor' => fn($cmd) => (new Zip())->run(),
                'message' => 'Zipping theme for staging',
            ],
            [
                'command' => $this->rsyncUp("themes/{$themeDir}.zip"),
                'executor' => fn($cmd) => exec($cmd),
                'message' => 'Pushing zip to staging',
            ],
            [
                'command' => $this->remoteCommand("rm -rf {$this->remoteContent("themes/{$themeDir}")}"),
                'executor' => fn($cmd) => exec($cmd),
                'message' => 'Removing old theme on staging',
            ],
            [
                'command' => $this->remoteCommand($this->wp("theme install {$this->remoteContent("themes/{$themeDir}.zip")} --force")),
                'executor' => fn($cmd) => exec($cmd),
                'message' => 'Installing theme on staging',
            ],
            [
                'command' => $this->remoteCommand("rm {$this->remoteContent("themes/{$themeDir}.zip")}"),
                'executor' => fn($cmd) => exec($cmd),
                'message' => 'Cleanup up remote zip file',
            ],
        ];

        foreach($commands as $command) {
            $this->runCommand(...$command);
        }
    }

    protected function pushDatabase(): void
    {
        $commands = [
            [
                'command' => WP_CONTENT_DIR . '/latest.sql',
                'executor' => fn($cmd) => file_exists($cmd) ? unlink($cmd) : '',
                'message' => 'Removing local sql file',
            ],
            [
                'command' => 'db export ' . $this->localContent('latest.sql'),
                'executor' => fn($cmd) => \WP_CLI::runcommand($cmd),
                'message' => 'Exporting local database',
            ],
            [
                'command' => $this->rsyncUp('latest.sql'),
                'executor' => fn($cmd) => exec($cmd),
                'message' => 'Pushing database to staging',
            ],
            [
                'command' => $this->remoteCommand($this->wp("db import {$this->remoteContent('latest.sql')}")),
                'executor' => fn($cmd) => exec($cmd),
                'message' => 'Importing database on staging',
            ],
            [
                'command' => $this->remoteCommand("rm {$this->remoteContent('latest.sql')}"),
                'executor' => fn($cmd) => exec($cmd),
                'message' => 'Removing sql file on staging',
            ],
            [
                'command' => $this->remoteCommand($this->wp("search-replace ". preg_replace("(^https?://)", "", get_home_url()) ." {$this->env()['STAGING_SITE_URL']}")),
                'executor' => fn($cmd) => exec($cmd),
                'message' => 'Replacing URLs in staging database',
            ],
        ];

        foreach($commands as $command) {
            $this->runCommand(...$command);
        }
    }
}
