<?php

namespace LeaseLeads\ThemeKit\Plugins\Providers;

use LeaseLeads\ThemeKit\Core\Contracts\ProviderInterface;
use LeaseLeads\ThemeKit\Core\Providers\AbstractProvider;
use LeaseLeads\ThemeKit\Core\Services\Filter;
use WP_Dependency_Installer;
use function LeaseLeads\ThemeKit\Core\Helpers\config;

class Plugins extends AbstractProvider implements ProviderInterface {


    public function boot(): void
    {
        add_action('after_switch_theme', function ($last_theme) {
            $this->installPlugins($last_theme);
        });
    }

    private function installPlugins($last_theme): void
    {
        $plugins = config('plugins.required');

        foreach ( $plugins as $key => $plugin_config ) {
            $plugin_slug = basename($plugin_config['slug'], '.php');

            $plugins[$key] = Filter::apply('plugins/required/' . $plugin_slug, $plugin_config);
        }

        WP_Dependency_Installer::instance(get_template_directory())
            ->register($plugins)
            ->run();

        $this->verifyRequiredPlugins($last_theme, $plugins);
    }

    private function verifyRequiredPlugins($last_theme, $plugins): void
    {
        foreach($plugins as $plugin) {

            if ( ! $plugin['required'] ) {
                continue;
            }

            if ( file_exists(WP_PLUGIN_DIR . '/' . $plugin['slug']) ) {
                continue;
            }

            add_action('admin_notices', function() use ($plugin) {
                $plugin_slug = basename($plugin['slug'], '.php');

                echo '<div class="notice notice-error"><p>' . sprintf(
                        Filter::apply("plugin/notice/fail/{$plugin_slug}", 'The required plugin %s could not be installed. Please install it to continue.'),
                        $plugin['name']
                    ) . '</p></div>';
            });

            $this->switchTheme($last_theme);
        }
    }

    /**
     * If the plugin is required and failed to install, we switch back to the previous theme and provide a
     * reason why we did so.
     */
    private function switchTheme($theme): void
    {
        update_option('template', $theme);
        update_option('stylesheet', $theme);
        update_option('current_theme', $theme);

        // Redirect to themes page to notify the user and avoid WordPress tripping up because the expected active
        // theme is not the current one.
        if ( wp_redirect(admin_url('themes.php')) ) {
            exit;
        }
    }
}
