<?php

namespace LeaseLeads\ThemeKit\Plugins\Services;



use LeaseLeads\ThemeKit\Core\Services\Filter;
use WP_Dependency_Installer;

class PluginLoader {

    protected $plugins = [];

    public function __construct() {
    }

    public function registerPlugin(array $configuration) : void {
        $this->plugins[] = new WpDependencyInstaller($configuration);
    }

    public function install(): void
    {
        add_action('after_switch_theme', function ($last_theme) {
            collect($this->plugins)
                ->each(fn(WpDependencyInstaller $plugin) => $plugin->install())
                ->each(fn(WpDependencyInstaller $plugin) => $plugin->activate());

            if ( ! $this->requiredPluginsInstalled() ) {
                $this->switchTheme($last_theme);
            };
        });
    }

    protected function requiredPluginsInstalled(): bool
    {
        $failed = collect($this->plugins)
            ->filter(fn (WpDependencyInstaller $installer) => ! $installer->isInstalled());

        if ( empty($failed) ) {
            return true;
        }

        $failed->each(fn($plugin) => $this->registerFailureNotice($plugin));

        return false;
    }

    protected function registerFailureNotice(array $plugin): void
    {
        add_action('admin_notices', function() use ($plugin) {
            $plugin_slug = basename($plugin['slug'], '.php');
            $message = sprintf('The required plugin %s could not be installed. Please install it to continue.', $plugin['name']);

            echo '<div class="notice notice-error"><p>' .
                Filter::apply("plugin/notice/fail/{$plugin_slug}", $message) .
                '</p></div>';
        });
    }

    /**
     * If any required plugins failed to install, we switch back to the previous theme and provide a
     * reason why we did so.
     */
    protected function switchTheme($theme): void
    {
        update_option('template', $theme);
        update_option('stylesheet', $theme);
        update_option('current_theme', $theme);

        // Redirect to themes page to notify the user and avoid WordPress tripping up because the expected active
        // theme is not the current one.
        if ( wp_redirect(admin_url('themes.php')) ) {
            exit;
        }
    }


}
