<?php

namespace LeaseLeads\ThemeKit\Data\Services;

use LeaseLeads\ThemeKit\Data\Contracts\DatableInterface;
use LeaseLeads\ThemeKit\Data\Contracts\DataInterface;
use LeaseLeads\ThemeKit\Fields\Services\Sanitizer;
use function LeaseLeads\ThemeKit\bound_by;


/**
 * Class FieldableDataAdapter
 */
class FieldableDataAdapter {

    /**
     * @template T of DataInterface
     * @param DatableInterface $datable
     * @param class-string<T> $dataClass
     */
    public function __construct(
            protected DatableInterface $datable,
            protected string $dataClass
    ) {
        if ( ! bound_by($dataClass, DataInterface::class) ) {
            throw new \InvalidArgumentException(
                sprintf(
                    'Expected data class of type %s, got %s',
                    $this->dataClass,
                    $dataClass
                )
            );
        }
    }

    public function getRawData(): array
    {
        return (new Sanitizer(
            $this->dataClass::sanitize($this->datable->getData()),
            $this->dataClass::defaults()
        ))->sanitize();
    }

    protected function getArgs() : array {
        $reflector = new \ReflectionClass($this->dataClass);
        $constructor = $reflector->getConstructor();
        $parameters = $constructor->getParameters();

        return collect($parameters)
            ->mapWithKeys(function($parameter, $key) {
                $key = $parameter->getName();
                $value = $this->getRawData()[$key] ?? null;

                if (is_array($value) && ! array_is_list($value) && $parameter->getType()?->getName() === 'object') {
                    $value = (object) $value;
                }

                if ($value === null && $parameter->isDefaultValueAvailable()) {
                    $value = $parameter->getDefaultValue();
                }

                return [$key => $value];
            })->toArray();
    }

    public function getData(): object
    {
        return new $this->dataClass(...$this->getArgs());
    }

}
