<?php

namespace LeaseLeads\ThemeKit\Data\Services;

use InvalidArgumentException;
use LeaseLeads\ThemeKit\Data\Contracts\DataInterface;

class Sanitizer {

    public static function make($data, $defaults): static
    {
        return new static($data, $defaults);
    }

    /**
     * @param array $data The user-provided data to sanitize.
     * @param array $defaults The default values and type/class mappings.
     */
    public function __construct(
        protected array $data,
        protected array $defaults
    ) {}

    /**
     * Transforms a scalar value based on the default value's type.
     *
     * @param mixed $value The input value to transform.
     * @param mixed $default The default value indicating the expected type.
     * @return mixed The transformed value or default value.
     */
    protected static function transformScalar(
        string|float|bool|int|null $value,
        string|float|bool|int|null $default) : mixed
    {
        if ($default === null) {
            return $value;
        }

        if ( $value === null ) {
            return $default;
        }

        if ( gettype($value) === gettype($default) ) {
            return $value;
        }

        $expectedType = gettype($default);

        switch ($expectedType) {
            case 'boolean':
                if (is_string($value)) {
                    $value = filter_var($value, FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE);
                }
                return is_bool($value) ? $value : $default;

            case 'integer':
                return is_numeric($value) ? (int) $value : $default;

            case 'string':
                return is_scalar($value) ? (string) $value : $default;

            case 'double':
                return is_numeric($value) ? (float) $value : $default;

            default:
                return $default;
        }
    }

    /**
     * Transforms a value into an object based on the specified class.
     *
     * @param mixed $value The input value to transform.
     * @param string $default The class name to transform into.
     * @return WP_Post|WP_Term|DataInterface|null The transformed object or null.
     * @throws InvalidArgumentException If the class does not exist.
     */
    protected static function transformObject(mixed $value, string $default): WP_Post|WP_Term|DataInterface|null {
        if (!class_exists($default)) {
            throw new InvalidArgumentException("Class $default does not exist.");
        }

        if ($value instanceof $default) {
            return $value;
        }

        if (is_array($value) && isset($value['ID'])) {
            $value = $value['ID'];
        } elseif (is_array($value) && isset($value['term_id'])) {
            $value = $value['term_id'];
        } elseif (is_numeric($value)) {
            $value = (int) $value;
        }

        if (in_array($default, ['WP_Post', 'WP_Term', 'WP_Comment', 'WP_User', 'WP_Widget'])) {
            $func = match($default) {
                'WP_Post' => 'get_post',
                'WP_Term' => 'get_term',
                'WP_Comment' => 'get_comment',
                'WP_User' => 'get_user',
            };

            $value = static::transformScalar($value, 0);

            if ($value === 0) {
                return null;
            }

            $item = $func($value);
            return ($item instanceof $default) ? $item : null;
        }

        if (class_implements($default, DataInterface::class)) {
            try {
                if (!is_array($value) || empty($value)) {
                    return $default::fromArray([]);
                }
                return $default::fromArray($value);
            } catch (InvalidArgumentException $e) {
                return null;
            }
        }

        return null;
    }

    protected function transformArray(mixed $value, array $default): array {
        if ( ! is_array($value) ) {
            $value = [];
        }

        if (empty($default)) {
            return $value;
        }

        return (new Sanitizer($value, $default))->sanitize();
    }

    protected function transformCallable(mixed $value, callable $default) : array
    {
        if (!is_array($value) || empty($value)) {
            $value = [];
        }

        $defaults = $default();

        return collect($value)
            ->map(function($item) use ($defaults) {
                return $this->sanitizeValue($item, $defaults);
            })
            ->all();
    }

    protected function transformString(mixed $value, string $default) : mixed {
        if (class_exists($default)) {
            return static::transformObject($value, $default);
        }

        return static::transformScalar($value, $default);
    }

    protected function sanitizeValue(mixed $value, mixed $default): mixed
    {
        $type = is_callable($default) ? 'callable' : gettype($default);

        return match($type) {
            'array' => $this->transformArray($value, $default),
            'callable' => $this->transformCallable($value, $default),
            'string' => $this->transformString($value, $default),
            'double', 'float', 'integer', 'boolean' => $this->transformScalar($value, $default),
            default => $value,
        };
    }

    /**
     * Sanitizes the input data based on the defaults.
     *
     * @return array The sanitized data.
     */
    public function sanitize(): array {
        $result = [];

        foreach ($this->defaults as $key => $default) {
            $result[$key] = $this->sanitizeValue($this->data[$key] ?? null, $default);
        }

        return $result;
    }
}
