<?php

namespace LeaseLeads\ThemeKit\Deploy\Commands;

use function LeaseLeads\ThemeKit\env;

class Publish extends Base {

    public function arguments() : array {
        return [
            'version' => 'Version type to tag the release with. One of "major", "minor", or "patch".',
        ];
    }

    public function name(): string
    {
        return 'publish';
    }

    public function description(): string
    {
        return 'Zips, versions, and deploys the theme to both git and the updates server.';
    }

    protected function bump($version, $type): string {
        $version = preg_replace('/[^0-9.]/', '', $version);
        $parts = explode('.', $version);

        switch ($type) {
            case 'major':
                $parts[0]++;
                $parts[1] = 0;
                $parts[2] = 0;
                break;
            case 'minor':
                $parts[1]++;
                $parts[2] = 0;
                break;
            case 'patch':
                $parts[2]++;
                break;
        }

        return implode('.', $parts);
    }

    public function run(): void
    {
        if ( empty($this->arguments['version']) ) {
            \WP_CLI::error('--version argument is required.');
            return;
        }

        $current = wp_get_theme()->get('Version');
        $next = $this->bump($current, $this->arguments['version']);

        $this->createRelease($next);
//        $this->deploy();

        \WP_CLI::success('Theme update has been published.');
    }

    protected function createRelease($version) {
        $changes = $this->getChangesSinceLastRelease();

        $commands = [
            [
                'command' => "git flow release start {$version}",
                'executor' => fn($cmd) => exec($cmd),
                'message' => 'Starting release branch',
            ],
            [
                'command' => "sed -i '' 's/Version: {$version}/Version: {$version}/' style.css",
                'executor' => fn($cmd) => exec($cmd),
                'message' => 'Updating version in style.css',
            ],
            [
                'command' => "git add style.css",
                'executor' => fn($cmd) => exec($cmd),
                'message' => 'Adding style.css to git',
            ],
            [
                'command' => "git commit -m 'version bump'",
                'executor' => fn($cmd) => exec($cmd),
                'message' => 'Committing version bump',
            ],
            [
                'command' => "git flow release finish -m 'Release {$version}' {$version}",
                'executor' => fn($cmd) => exec($cmd),
                'message' => 'Finishing release branch',
            ]
        ];
    }

    protected function getChangesSinceLastRelease(): string {
        $lastTag = exec('git describe --tags --abbrev=0');
        $changes = exec("git log {$lastTag}..HEAD --pretty=format:'%h %s'");

        return $changes;
    }

    // Sync plugins and uploads from staging using rsync
    protected function pullFiles(): void
    {
        $commands = [
            [
                'command' => $this->rsyncDown('plugins/'),
                'executor' => fn($cmd) => exec($cmd),
                'message' => 'Pulling plugins from staging',
            ],
            [
                'command' => $this->rsyncDown('uploads/'),
                'executor' => fn($cmd) => exec($cmd),
                'message' => 'Pulling uploads from staging',
            ]
        ];

        foreach($commands as $command) {
            $this->runCommand(...$command);
        }
    }

    protected function pullDatabase(): void
    {
        $vars = $this->env();

        $local = WP_CONTENT_DIR . '/latest.sql';
        $remote = $vars['STAGING_SITE_PATH'] . '/wp-content/latest.sql';

        $commands = [
            [
                'command' => WP_CONTENT_DIR . '/latest.sql',
                'executor' => fn($cmd) => file_exists($cmd) ? unlink($cmd) : '',
                'message' => 'Removing local sql file if it exists',
            ],
            [
                'command' => $this->remoteCommand($this->wp("db export {$remote}", '--all-tables')),
                'executor' => fn($cmd) => exec($cmd),
                'message' => 'Exporting database on staging',
            ],
            [
                'command' => $this->rsyncDown('latest.sql'),
                'executor' => fn($cmd) => exec($cmd),
                'message' => 'Copying database to local',
            ],
            [
                'command' => $this->remoteCommand("rm {$remote}"),
                'executor' => fn($cmd) => exec($cmd),
                'message' => 'Removing sql files on staging',
            ],
            [
                'command' => $this->wp("db import {$local}", '', 'local'),
                'executor' => fn($cmd) => \WP_CLI::runcommand($cmd),
                'message' => 'Importing database on local',
            ],
            [
                'command' => 'If "ERROR at line 1 in file..." appears above, it can be ignored.',
                'executor' => fn($cmd) => '',
                'message' => \WP_CLI::colorize('%y⚠ Note...%n'),
            ],
            [
                'command' => $this->wp('search-replace ' . $this->env()['STAGING_SITE_URL'] . ' ' . preg_replace("(^https?://)", "", get_home_url()), '', 'local'),
                'executor' => fn($cmd) => \WP_CLI::runcommand($cmd),
                'message' => 'Updating URLs in local database',
            ],
            [
                'command' => WP_CONTENT_DIR . '/latest.sql',
                'executor' => fn($cmd) => file_exists($cmd) ? unlink($cmd) : '',
                'message' => 'Removing local sql file',
            ],
        ];

        foreach($commands as $command) {
            $this->runCommand(...$command);
        }
    }
}
