<?php

namespace LeaseLeads\ThemeKit\Deploy\Commands;

use function LeaseLeads\ThemeKit\env;

class Pull extends BaseCommand {

    public function name(): string
    {
        return 'pull';
    }

    public function description(): string
    {
        return 'Pulls database, plugins, and uploads down from staging to local.';
    }

    public function run(): void
    {
        $this->pullFiles();
        $this->pullDatabase();

        \WP_CLI::success('Database, plugins, and uploads have been pulled from staging.');
    }

    // Sync plugins and uploads from staging using rsync
    protected function pullFiles(): void
    {
        $commands = [
            [
                'command' => $this->rsyncDown('plugins/'),
                'executor' => fn($cmd) => exec($cmd),
                'message' => 'Pulling plugins from staging',
            ],
            [
                'command' => $this->rsyncDown('uploads/'),
                'executor' => fn($cmd) => exec($cmd),
                'message' => 'Pulling uploads from staging',
            ]
        ];

        foreach($commands as $command) {
            $this->runCommand(...$command);
        }
    }

    protected function pullDatabase(): void
    {
        $vars = $this->env();

        $local = WP_CONTENT_DIR . '/latest.sql';
        $remote = $vars['STAGING_SITE_PATH'] . '/wp-content/latest.sql';

        $commands = [
            [
                'command' => WP_CONTENT_DIR . '/latest.sql',
                'executor' => fn($cmd) => file_exists($cmd) ? unlink($cmd) : '',
                'message' => 'Removing local sql file if it exists',
            ],
            [
                'command' => $this->remoteCommand($this->wp("db export {$remote}", '--all-tables')),
                'executor' => fn($cmd) => exec($cmd),
                'message' => 'Exporting database on staging',
            ],
            [
                'command' => $this->rsyncDown('latest.sql'),
                'executor' => fn($cmd) => exec($cmd),
                'message' => 'Copying database to local',
            ],
            [
                'command' => $this->remoteCommand("rm {$remote}"),
                'executor' => fn($cmd) => exec($cmd),
                'message' => 'Removing sql files on staging',
            ],
            [
                'command' => $this->wp("db import {$local}", '', 'local'),
                'executor' => fn($cmd) => \WP_CLI::runcommand($cmd),
                'message' => 'Importing database on local',
            ],
            [
                'command' => 'If "ERROR at line 1 in file..." appears above, it can be ignored.',
                'executor' => fn($cmd) => '',
                'message' => \WP_CLI::colorize('%y⚠ Note...%n'),
            ],
            [
                'command' => $this->wp('search-replace ' . $this->env()['STAGING_SITE_URL'] . ' ' . preg_replace("(^https?://)", "", get_home_url()), '', 'local'),
                'executor' => fn($cmd) => \WP_CLI::runcommand($cmd),
                'message' => 'Updating URLs in local database',
            ],
            [
                'command' => WP_CONTENT_DIR . '/latest.sql',
                'executor' => fn($cmd) => file_exists($cmd) ? unlink($cmd) : '',
                'message' => 'Removing local sql file',
            ],
        ];

        foreach($commands as $command) {
            $this->runCommand(...$command);
        }
    }
}
