<?php

namespace LeaseLeads\ThemeKit\Fields\Data;


use LeaseLeads\ThemeKit\Data\Contracts\DataInterface;
use LeaseLeads\ThemeKit\Fields\Services\Sanitizer;

class ColorSelect extends BaseData implements DataInterface {

    public static function defaults() : array
    {
        return [
            'color' => null,
            'label' => null,
        ];
    }

    public static function sanitize(mixed $data): array
    {
        if ( is_string($data) ) {
            $data = [
                'color' => !str_starts_with($data, 'color') ? $data : null,
                'label' => str_starts_with($data, 'color') ? $data : null,
            ];
        }

        if ( is_array($data) ) {
            $data = [
                'color' => $data['color'] ?? null,
                'label' => $data['label'] ?? null,
            ];
        }

        if ( is_null($data) ) {
            $data = static::defaults();
        }

        return (new Sanitizer($data, static::defaults()))
            ->sanitize();
    }

    public function __construct(
        public ?string $color = null,
        public ?string $label = null
    ) {}

    public function __toString(): string
    {

        if ( ! $this->hasColor() ) {
            return '';
        }

        if ( empty($this->label) ) {
            return $this->color;
        }

        if (! str_starts_with( strtolower($this->label), 'color') ) {
            return $this->color;
        }

        return 'var(--' . sanitize_title($this->label) . ')';
    }

    public function hasColor(): bool
    {
        return ! empty($this->color) || ! empty($this->label);
    }

    public function withDefault(string $color = 'rgba(255, 255, 255, 1)') : static {
        if ( empty($this->color) ) {
            $this->color = $color;
        }

        return $this;
    }

    public function asVariable(string $name) : string {
        if ( ! $this->hasColor() ) {
            return '';
        }

        return sprintf(
            '--%s: %s;',
            $name,
            (string) $this
        );
    }

    public function raw() : ?string {
        return $this->color;
    }
}
