<?php

namespace LeaseLeads\ThemeKit\Plugins\Services;



use LeaseLeads\ThemeKit\Core\Services\Filter;
use WP_Dependency_Installer;

class PluginLoader {

    protected $plugins = [];


    public function __construct() {
    }

    public function registerPlugin(array $configuration) : void {
        $this->plugins[] = new WpDependencyInstaller($configuration);
    }

    public function install(): void
    {
        collect($this->plugins)
            ->each(fn(WpDependencyInstaller $plugin) => $plugin->install())
            ->each(fn(WpDependencyInstaller $plugin) => $plugin->activate());

        add_action('admin_init', function() {
            if ( ! $this->requiredPluginsInstalled() ) {
                $this->switchTheme();
            };
        });
    }

    protected function requiredPluginsInstalled(): bool
    {
        $failed = collect($this->plugins)
            ->filter(fn (WpDependencyInstaller $installer) => ! $installer->isInstalled());

        if ( $failed->isEmpty() ) {
            return true;
        }

        $failed->each(fn($plugin) => $this->registerFailureNotice($plugin));

        return false;
    }

    protected function registerFailureNotice(WpDependencyInstaller $plugin): void
    {
        add_action('admin_notices', function() use ($plugin) {
            $plugin_slug = basename($plugin->configuration()['slug'], '.php');
            $message = sprintf('The required plugin %s could not be installed. Please install it to continue.', $plugin->configuration()['name']);

            echo '<div class="notice notice-error"><p>' .
                Filter::apply("plugin/notice/fail/{$plugin_slug}", $message) .
                '</p></div>';
        });
    }

    /**
     * If any required plugins failed to install, we switch back to the previous theme and provide a
     * reason why we did so.
     */
    protected function switchTheme(): void
    {
        add_action('admin_notices', function() {
            $message = 'Some required plugins could not be installed by the theme. Switching back to the default WordPress theme.';

            echo '<div class="notice notice-warning"><p>' . $message . '</p></div>';
        });

        add_action('shutdown', function() use ($theme) {
            switch_theme( WP_DEFAULT_THEME );
        });
    }


}
